-- Author:BayernGamers
-- Name:RealGPS Generate Vehicle XML
-- Namespace: local
-- Description: A Script that generates the RealGPS Vehicle XML Entries for the Default Vehicles
-- Icon:
-- Hide: no
-- AlwaysLoaded: no

function math.round(x, decimals)
    return math.floor(x * math.pow(10, decimals or 0) + 0.5) / math.pow(10, decimals or 0)
end

function fixNegativeZero(value)
    if value == 0 and tostring(value) == "-0" then
        return 0
    end
    return value
end

local function getIndexPath(node)
    local name = getName(node)
    local index = getChildIndex(node)
    local parent = getParent(node)
    local grandParent = nil
    if parent ~= 0 then
        grandParent = getParent(parent)
    end
    if grandParent == getRootNode() then
        local p_name, p_index = getIndexPath(parent)
        name = p_name .. name
        index = p_index .. index
    elseif parent ~= 0 and parent ~= getRootNode() then
        local p_name, p_index = getIndexPath(parent)
        name = p_name .. "|" .. name
        index = p_index .. "|" .. index
    else
        name = name .. ">"
        index = index .. ">"

    end
    return name, index
end

function createEntry()
    if getNumSelected() == 0 then
        print("Error: Please select the realGPS transform group and place it in the vehicle. Scale/Position/Rotate the transform group to the desired position and size. Then select the transform group and the node to be hidden (optional). Then click the button.")
        return
    end

    if getNumSelected() > 2 then
        print("Error: Please only select one transform group and the hideNode.")
        return
    end

    local selectedObject = getSelection(0)
    if getHasClassId(selectedObject, ClassIds.TRANSFORM_GROUP) == false then
        print("Error: Please select a transform group.")
        return
    end

    local hideNodeIndexPath = ""
    if getNumSelected() == 2 then
        local hideNode = getSelection(1)
        
        _, hideNodeIndexPath = getIndexPath(hideNode)
        hideNodeIndexPath = " hideNode=\"" .. hideNodeIndexPath .. "\""
    end

    local _, indexPath = getIndexPath(selectedObject)
    local scaleXraw, scaleYraw, scaleZraw = getScale(selectedObject)
    local scaleX = fixNegativeZero(math.round(scaleXraw, 3))
    local scaleY = fixNegativeZero(math.round(scaleYraw, 3))
    local scaleZ = fixNegativeZero(math.round(scaleZraw, 3))

    local aspectRatio = math.round(scaleX / scaleZ, 3)

    local backgroundZoomMin = 1
    if aspectRatio < 1 then
        backgroundZoomMin = math.round(1.35 * (1 /aspectRatio), 2)
    else
        backgroundZoomMin = math.round(1.35 * aspectRatio, 2)
    end

    local plane = getChildAt(selectedObject, 0)
    setVisibility(plane, false)

    local planeScaleX, planeScaleY, planeScaleZ = getScale(plane)

    if planeScaleX ~= 1 or planeScaleY ~= 1 or planeScaleZ ~= 1 then
        print("Error: Please reset the scale of the plane to 1. Scale the transform group instead.")
        return
    end

    local vehicleEntry = [[
    <realGPS isMotorActiveDependent="false" isIgnitionDependent="true">
        <controls allow="true" allowRotation="true" allowZoom="true" />
        <map linkNode="]] .. indexPath .. [["]] .. hideNodeIndexPath .. [[ intensity="0.2" idleValue="-1">
            <projection mode="PLAYER" playerSize="0.075" playerSizeMin="0.05" playerSizeMax="0.075" backgroundZoom="1" backgroundZoomMax="20" backgroundZoomMin="]] .. backgroundZoomMin .. [[" aspectRatio="]] .. aspectRatio .. [["/>
        </map>
    </realGPS>
    ]]
    print("\n== Vehicle XML Entry =================\n")
    print(vehicleEntry)
    print("\n====================================")
end

function restorePlane()
    if getNumSelected() == 0 then
        print("Error: Please select the realGPS transform group to restore the Plane.")
        return
    end

    if getNumSelected() > 1 then
        print("Error: Please only select one transform group.")
        return
    end

    local selectedObject = getSelection(0)
    if getHasClassId(selectedObject, ClassIds.TRANSFORM_GROUP) == false then
        print("Error: Please select a transform group.")
        return
    end

    local path = getEditorDirectory() .. "scripts/realGPSSetup/realGPSTemplate.i3d"
    local node = loadI3DFile(path, false, false, true)
    local template = getChildAt(node, 0)
    local plane = getChildAt(template, 0)
    link(selectedObject, plane)
end

function deletePlane()
    if getNumSelected() == 0 then
        print("Error: Please select the realGPS transform group to restore the Plane.")
        return
    end

    if getNumSelected() > 1 then
        print("Error: Please only select one transform group.")
        return
    end

    local selectedObject = getSelection(0)
    if getHasClassId(selectedObject, ClassIds.TRANSFORM_GROUP) == false then
        print("Error: Please select a transform group.")
        return
    end

    local plane = getChildAt(selectedObject, 0)
    delete(plane)
end

function showPlane()
    if getNumSelected() == 0 then
        print("Error: Please select the realGPS transform group to restore the Plane.")
        return
    end

    if getNumSelected() > 1 then
        print("Error: Please only select one transform group.")
        return
    end

    local selectedObject = getSelection(0)
    if getHasClassId(selectedObject, ClassIds.TRANSFORM_GROUP) == false then
        print("Error: Please select a transform group.")
        return
    end

    local plane = getChildAt(selectedObject, 0)
    setVisibility(plane, true)
end

local windowSizer = UIRowLayoutSizer.new()
local realGPSWindow = UIWindow.new(windowSizer, "RealGPS Vehicle Setup")

local rowLayoutSizer = UIRowLayoutSizer.new()
UIPanel.new(windowSizer, rowLayoutSizer)

local rowSizer = UIRowLayoutSizer.new()
UIPanel.new(rowLayoutSizer, rowSizer, -1, -1, 400, -1, BorderDirection.ALL, 10, 1)

UIButton.new(rowSizer, "Create Vehicle XML Entry", createEntry)
UIButton.new(rowSizer, "Show plane", showPlane)
UIButton.new(rowSizer, "Delete plane", deletePlane)
UIButton.new(rowSizer, "Restore plane", restorePlane)
realGPSWindow:showWindow()


local path = getEditorDirectory() .. "scripts/realGPSSetup/realGPSTemplate.i3d"
local node = loadI3DFile(path, false, false, true)
local template = getChildAt(node, 0)
link(getRootNode(), template)