--[[
    RealGPS

    Shows the current GPS position of the player on a plane in the vehicle.

	@author: 		BayernGamers
	@date: 			11.03.2025
	@version:		2.0

	History:		v1.0 @15.07.2023 - initial implementation in FS 22
                    ------------------------------------------------------------------------------------------------------
                    v1.1 @16.07.2023 - update for Multi-PDA Support
                    ------------------------------------------------------------------------------------------------------
                    v1.2 @17.07.2023 - added glowShader to PDA
                    ------------------------------------------------------------------------------------------------------
                    v2.0 @11.03.2025 - convert and re-write for FS25
                    ------------------------------------------------------------------------------------------------------
                    v2.1 @15.06.2025 - added support for Dashboard Compounds and multiple PDA's
                    ------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/SetupModeEvent.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/RotationModeEvent.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/MapZoomEvent.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "RealGPS.lua")

RealGPS = {}
RealGPS.MOD_DIR = g_currentModDirectory
RealGPS.MOD_NAME = g_currentModName
RealGPS.MOD_SETTINGS_DIR = g_currentModSettingsDirectory
RealGPS.BASE_PATH = "vehicle.realGPS"
RealGPS.CONTROLS_BASE_PATH = RealGPS.BASE_PATH .. ".controls"
RealGPS.MAP_BASE_PATH = RealGPS.BASE_PATH .. ".map(?)"
RealGPS.PROJECTION_BASE_PATH = RealGPS.MAP_BASE_PATH .. ".projection"
RealGPS.HIDE_NODES_BASE_PATH = RealGPS.BASE_PATH .. ".hideNodes.hideNode(?)"
RealGPS.SHOW_NODES_BASE_PATH = RealGPS.BASE_PATH .. ".showNodes.showNode(?)"

RealGPS.defaultVehicles = {}
RealGPS.DEFAULT_VEHICLE_BASE_PATH = "defaultVehicles.vehicle(?)"
RealGPS.DEFAULT_VEHICLE_XML = RealGPS.MOD_DIR .. "xml/defaultVehicles.xml"
RealGPS.DEFAULT_VEHICLE_SCHEMA = XMLSchema.new("realGPSDefaultVehicles")

RealGPS.DASHBOARD_COMPOUND_BASE_PATH = "dashboardCompounds.dashboardCompound(?).realGPS(?)"

createFolder(RealGPS.MOD_SETTINGS_DIR)

function RealGPS.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Dashboard, specializations) and SpecializationUtil.hasSpecialization(Enterable, specializations) and not SpecializationUtil.hasSpecialization(Locomotive, specializations)
end

function RealGPS.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("RealGPS")

    g_vehicleConfigurationManager:addConfigurationType("realGPS", g_i18n:getText("configuration_realGPS"), nil, VehicleConfigurationItem)

    schema:register(XMLValueType.BOOL, RealGPS.BASE_PATH .. "#isMotorActiveDependent", "Disable PDA when motor is off", false)
    schema:register(XMLValueType.BOOL, RealGPS.BASE_PATH .. "#isIgnitionDependent", "Disable PDA when the ignition is off", false)

    schema:register(XMLValueType.BOOL, RealGPS.CONTROLS_BASE_PATH .. "#allow", "Allow the player to control the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.CONTROLS_BASE_PATH .. "#allowRotation", "Allow the player to rotate the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.CONTROLS_BASE_PATH .. "#allowZoom", "Allow the player to zoom the PDA", true)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.MAP_BASE_PATH .. "#linkNode", "Node to load the PDA into")
    schema:register(XMLValueType.NODE_INDEX, RealGPS.MAP_BASE_PATH .. "#hideNode", "Node to hide the original Dashboards")
    schema:register(XMLValueType.NODE_INDEX, RealGPS.MAP_BASE_PATH .. "#showNode", "Node which will be set to visible when RealGPS is active")
    schema:register(XMLValueType.FLOAT, RealGPS.MAP_BASE_PATH .. "#intensity", "Intensity of the glow shader", 1.0)
    schema:register(XMLValueType.FLOAT, RealGPS.MAP_BASE_PATH .. "#idleValue", "Value to set the glow shader to when the PDA is not active", -1)
    schema:register(XMLValueType.BOOL, RealGPS.MAP_BASE_PATH .. "#toggleVisibility", "Toggle the visibility of the PDA instead of the idleValue", false)
    schema:register(XMLValueType.FLOAT, RealGPS.MAP_BASE_PATH .. "#stateDelay", "Delay before the PDA is shown", 0.0)

    schema:register(XMLValueType.STRING, RealGPS.PROJECTION_BASE_PATH .. "#mode", "Projection mode of the PDA", "PLAYER")
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#playerSize", "Size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#playerSizeMax", "Max size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#playerSizeMin", "Min size of the player marker", 0.001)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#backgroundZoom", "Zoom of the background", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#backgroundZoomMax", "Max zoom of the background", 20)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#backgroundZoomMin", "Min zoom of the background", 1.35)
    schema:register(XMLValueType.FLOAT, RealGPS.PROJECTION_BASE_PATH .. "#aspectRatio", "Aspect ratio of the PDA", 1)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.HIDE_NODES_BASE_PATH .. "#node", "Node to hide when RealGPS is configured")
    schema:register(XMLValueType.BOOL, RealGPS.HIDE_NODES_BASE_PATH .. "#isDashboard", "Is the node a dashboard", false)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.SHOW_NODES_BASE_PATH .. "#node", "Node to show when RealGPS is configured")

    Dashboard.registerDashboardXMLPaths(schema, "vehicle.realGPS.dashboards", {"playerPositionX", "playerPositionZ", "playerRotation", "gpsWorkingWidth"})
    Dashboard.registerDashboardXMLPaths(Dashboard.compoundsXMLSchema, "dashboardCompounds.dashboardCompound(?).realGPS.dashboards", {"playerPositionX", "playerPositionZ", "playerRotation", "gpsWorkingWidth"})
    Dashboard.registerDashboardXMLPaths(Dashboard.compoundsXMLSchema, "dashboardCompounds.dashboardCompound(?).configuration(?).realGPS.dashboards", {"playerPositionX", "playerPositionZ", "playerRotation", "gpsWorkingWidth"})

    schema:setXMLSpecializationType()

    -- Dashboard Compounds
    schema = Dashboard.compoundsXMLSchema
    schema:setXMLSpecializationType("RealGPSDasboardCompound")

    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. "#isMotorActiveDependent", "Disable PDA when motor is off", false)
    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. "#isIgnitionDependent", "Disable PDA when the ignition is off", false)

    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".controls#allow", "Allow the player to control the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".controls#allowRotation", "Allow the player to rotate the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".controls#allowZoom", "Allow the player to zoom the PDA", true)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#linkNode", "Node to load the PDA into")
    schema:register(XMLValueType.NODE_INDEX, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#hideNode", "Node to hide the original Dashboards")
    schema:register(XMLValueType.NODE_INDEX, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#showNode", "Node which will be set to visible when RealGPS is active")
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#intensity", "Intensity of the glow shader", 1.0)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#idleValue", "Value to set the glow shader to when the PDA is not active", -1)
    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#toggleVisibility", "Toggle the visibility of the PDA instead of the idleValue", false)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?)#stateDelay", "Delay before the PDA is shown", 0.0)

    schema:register(XMLValueType.STRING, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#mode", "Projection mode of the PDA", "PLAYER")
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#playerSize", "Size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#playerSizeMax", "Max size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#playerSizeMin", "Min size of the player marker", 0.001)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#backgroundZoom", "Zoom of the background", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#backgroundZoomMax", "Max zoom of the background", 20)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#backgroundZoomMin", "Min zoom of the background", 1.35)
    schema:register(XMLValueType.FLOAT, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".map(?).projection#aspectRatio", "Aspect ratio of the PDA", 1)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".hideNodes.hideNode(?)#node", "Node to hide when RealGPS is configured")
    schema:register(XMLValueType.BOOL, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".hideNodes.hideNode(?)#isDashboard", "Is the node a dashboard", false)

    schema:register(XMLValueType.NODE_INDEX, RealGPS.DASHBOARD_COMPOUND_BASE_PATH .. ".showNodes.showNode(?)#node", "Node to show when RealGPS is configured")

    -- Default Vehicles
    schema = RealGPS.DEFAULT_VEHICLE_SCHEMA
    schema:setXMLSpecializationType("RealGPSDefaultVehicle")

    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. "#filename", "Filename of the vehicle", nil)
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. "#isMotorActiveDependent", "Disable PDA when motor is off", false)
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. "#isIgnitionDependent", "Disable PDA when the ignition is off", false)

    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".controls#allow", "Allow the player to control the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".controls#allowRotation", "Allow the player to rotate the PDA", true)
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".controls#allowZoom", "Allow the player to zoom the PDA", true)

    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#rootNode", "Node to load the PDA into")
    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#hideNode", "Node to hide the original Dashboards")
    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#showNode", "Node which will be set to visible when RealGPS is active")
    schema:register(XMLValueType.VECTOR_TRANS, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#translation", "Translation of the PDA")
    schema:register(XMLValueType.VECTOR_ROT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#rotation", "Rotation of the PDA")
    schema:register(XMLValueType.VECTOR_SCALE, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#scale", "Scale of the PDA")

    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#intensity", "Intensity of the glow shader", 1.0)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#idleValue", "Value to set the glow shader to when the PDA is not active", -1)
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#toggleVisibility", "Toggle the visibility of the PDA instead of the idleValue", false)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?)#stateDelay", "Delay before the PDA is shown", 0.0)

    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#mode", "Projection mode of the PDA", "PLAYER")
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#playerSize", "Size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#playerSizeMax", "Max size of the player marker", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#playerSizeMin", "Min size of the player marker", 0.001)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#backgroundZoom", "Zoom of the background", 1)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#backgroundZoomMax", "Max zoom of the background", 20)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#backgroundZoomMin", "Min zoom of the background", 1.35)
    schema:register(XMLValueType.FLOAT, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".map(?).projection#aspectRatio", "Aspect ratio of the PDA", 1)

    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".hideNodes.hideNode(?)#node", "Node to hide when RealGPS is configured")
    schema:register(XMLValueType.BOOL, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".hideNodes.hideNode(?)#isDashboard", "Is the node a dashboard", false)

    schema:register(XMLValueType.STRING, RealGPS.DEFAULT_VEHICLE_BASE_PATH .. ".showNodes.showNode(?)#node", "Node to show when RealGPS is configured")

    schema:setXMLSpecializationType()
end

function RealGPS.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onPreInitComponentPlacement", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterDashboardValueTypes", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", RealGPS)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", RealGPS)

	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", RealGPS)
end

function RealGPS.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadGPSMap", RealGPS.loadGPSMap)
    SpecializationUtil.registerFunction(vehicleType, "postLoadGPSMap", RealGPS.postLoadGPSMap)
    SpecializationUtil.registerFunction(vehicleType, "setMapZoom", RealGPS.setMapZoom)
    SpecializationUtil.registerFunction(vehicleType, "setPlayerAndMapZoom", RealGPS.setPlayerAndMapZoom)
    SpecializationUtil.registerFunction(vehicleType, "setPlayerSize", RealGPS.setPlayerSize)
    SpecializationUtil.registerFunction(vehicleType, "setRotateMap", RealGPS.setRotateMap)
    SpecializationUtil.registerFunction(vehicleType, "setMotorDependentVisibility", RealGPS.setMotorDependentVisibility)
    SpecializationUtil.registerFunction(vehicleType, "calculateMapAndPlayerPosition", RealGPS.calculateMapAndPlayerPosition)
    SpecializationUtil.registerFunction(vehicleType, "calculateRootWorldYAxisRotation", RealGPS.calculateRootWorldYAxisRotation)
    SpecializationUtil.registerFunction(vehicleType, "mapRange", RealGPS.mapRange)

    -- RealWorkCameras also registers this function, no need to double functionality here
    if vehicleType.functions["getIsIgnitionActive"] == nil then
        SpecializationUtil.registerFunction(vehicleType, "getIsIgnitionActive", RealGPS.getIsIgnitionActive)
    end

    SpecializationUtil.registerFunction(vehicleType, "loadDefaultVehicles", RealGPS.loadDefaultVehicles)
    SpecializationUtil.registerFunction(vehicleType, "resetDelayTimer", RealGPS.resetDelayTimer)
    SpecializationUtil.registerFunction(vehicleType, "loadDashboardCompoundRealGPS", RealGPS.loadDashboardCompoundRealGPS)

    if vehicleType.functions["removeDashboardByNode"] == nil then
        SpecializationUtil.registerFunction(vehicleType, "removeDashboardByNode", RealGPS.removeDashboardByNode)
    end
end

function RealGPS.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "onDashboardCompoundLoaded", RealGPS.onDashboardCompoundLoaded)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "loadDashboardsFromXML", RealGPS.loadDashboardsFromXML)
end

function RealGPS:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self.spec_realGPS

    if spec.hasRealGPSMap == true then
        
        for i, map in ipairs(spec.maps) do
            local mapKey = key .. ".map(" .. (i - 1) .. ")"
            setXMLFloat(xmlFile.handle, mapKey .. "#mapZoom", map.zoom)
            setXMLFloat(xmlFile.handle, mapKey .. "#playerSize", map.projection.playerSize)
            setXMLBool(xmlFile.handle, mapKey .. "#rotateMap", map.rotatingMap)
            log:printDevInfo(string.format("Saved RealGPS vehicle settings for vehicle %s to savegame XML file", self:getName()), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Map Index: " .. tostring(i), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Savegame XML Key: " .. mapKey, LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Map Zoom: " .. tostring(map.zoom), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Player Size: " .. tostring(map.projection.playerSize), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Rotate Map: " .. tostring(map.rotatingMap), LoggingUtil.DEBUG_LEVELS.HIGH)
        end
	end
end

function RealGPS:onReadStream(streamId, connection)
    local spec = self.spec_realGPS

	if spec.hasRealGPSMap == true then
        spec.isInSetupMode = streamReadBool(streamId)

        for i, map in ipairs(spec.maps) do
            self:setMapZoom(map, streamReadFloat32(streamId))
            self:setPlayerSize(map, streamReadFloat32(streamId))
            self:setRotateMap(map, streamReadBool(streamId))
            log:printDevInfo(string.format("Read RealGPS vehicle settings for vehicle %s from multiplayer sync", self:getName()), LoggingUtil.DEBUG_LEVELS.HIGH)
        end
	end
end

function RealGPS:onWriteStream(streamId, connection)
    local spec = self.spec_realGPS

	if spec.hasRealGPSMap == true then
        streamWriteBool(streamId, spec.isInSetupMode)

        for i, map in ipairs(spec.maps) do
            streamWriteFloat32(streamId, map.zoom)
            streamWriteFloat32(streamId, map.projection.playerSize)
            streamWriteBool(streamId, map.rotatingMap)
            log:printDevInfo(string.format("Wrote RealGPS vehicle settings for vehicle %s for multiplayer sync", self:getName()), LoggingUtil.DEBUG_LEVELS.HIGH)
        end
	end
end

function RealGPS:onPreLoad(savegame)
    self.spec_realGPS = {}
    local spec = self.spec_realGPS
end

function RealGPS:onLoad(savegame)
    local spec = self.spec_realGPS

    spec.hasRealGPSMap = false
    spec.selectedMapIndex = 1
    spec.maps = {}
    spec.actionEvents = {}
    spec.hideNodes = {}
    spec.showNodes = {}

    log:printDevInfo(string.format("RealGPS config Id for vehicle %s is: '%s'", self:getName(), tostring(self.configurations["realGPS"])), LoggingUtil.DEBUG_LEVELS.HIGH)
    if self.configurations["realGPS"] ~= nil and self.configurations["realGPS"] == 2 then
        self:loadDefaultVehicles()

        local dlcFilename = nil
        if self.xmlFile:getFilename():find("pdlc") ~= nil then
            local pattern = ".*pdlc"
            dlcFilename = "pdlc" .. self.xmlFile:getFilename():gsub(pattern, "")
        end

        if (self.xmlFile:getFilename() ~= nil and RealGPS.defaultVehicles[self.xmlFile:getFilename()] ~= nil) or (dlcFilename ~= nil and RealGPS.defaultVehicles[dlcFilename] ~= nil) then
            local defaultVehicle = nil

            if dlcFilename ~= nil then
                defaultVehicle = RealGPS.defaultVehicles[dlcFilename]
            else
                defaultVehicle = RealGPS.defaultVehicles[self.xmlFile:getFilename()]
            end

            spec.isIgnitionDependent = defaultVehicle.isIgnitionDependent
            spec.isMotorActiveDependent = defaultVehicle.isMotorActiveDependent

            if spec.isIgnitionDependent == true and spec.isMotorActiveDependent == true then
                spec.isMotorActiveDependent = false
            end

            spec.isLoading = true

            spec.controls = {}
            spec.controls.allow = defaultVehicle.controls.allow
            spec.controls.allowRotation = defaultVehicle.controls.allowRotation
            spec.controls.allowZoom = defaultVehicle.controls.allowZoom
            

            for _, hideNode in ipairs(defaultVehicle.hideNodes) do
                local node = I3DUtil.indexToObject(self.components, hideNode.node, self.i3dMappings)
                if node ~= nil then
                    table.insert(spec.hideNodes, {node = node, isDashboard = hideNode.isDashboard})
                end
            end

            for _, showNode in ipairs(defaultVehicle.showNodes) do
                local node = I3DUtil.indexToObject(self.components, showNode, self.i3dMappings)
                if node ~= nil then
                    table.insert(spec.showNodes, node)
                end
            end


            for i, map in ipairs(defaultVehicle.maps) do
                local linkNode = createTransformGroup("realGPSMap")
                local rootNode = I3DUtil.indexToObject(self.components, map.rootNode, self.i3dMappings)

                link(rootNode, linkNode)
                setTranslation(linkNode, unpack(map.translation))
                setRotation(linkNode, unpack(map.rotation))
                setScale(linkNode, unpack(map.scale))

                map.linkNode = linkNode

                if map.hideNode ~= nil then
                    local hideNode = I3DUtil.indexToObject(self.components, map.hideNode, self.i3dMappings)
                    setVisibility(hideNode, false)
                end

                if map.showNode ~= nil then
                    local showNode = I3DUtil.indexToObject(self.components, map.showNode, self.i3dMappings)
                    setVisibility(showNode, true)
                end

                spec.isInSetupMode = false

                map.gpsAvailable = false
                self:resetDelayTimer(map)

                table.insert(spec.maps, map)
                if g_currentMission:getIsClient() then
                    self:loadGPSMap(map)
                end
            end
        end


        if self.xmlFile:hasProperty(RealGPS.BASE_PATH) and not spec.hasRealGPSMap then
            spec.isIgnitionDependent = self.xmlFile:getValue(RealGPS.BASE_PATH .. "#isIgnitionDependent", false)
            spec.isMotorActiveDependent = self.xmlFile:getValue(RealGPS.BASE_PATH .. "#isMotorActiveDependent", false)

            if spec.isIgnitionDependent == true and spec.isMotorActiveDependent == true then
                spec.isMotorActiveDependent = false
            end

            spec.isLoading = true

            spec.controls = {}
            spec.controls.allow = self.xmlFile:getValue(RealGPS.CONTROLS_BASE_PATH .. "#allow", true)
            spec.controls.allowRotation = self.xmlFile:getValue(RealGPS.CONTROLS_BASE_PATH .. "#allowRotation", true)
            spec.controls.allowZoom = self.xmlFile:getValue(RealGPS.CONTROLS_BASE_PATH .. "#allowZoom", true)

            spec.isInSetupMode = false

            self.xmlFile:iterate(RealGPS.MAP_BASE_PATH:sub(1, -4), function(index, key)
                local map = {}
                map.linkNode = self.xmlFile:getValue(key .. "#linkNode", nil, self.components, self.i3dMappings)
                map.intensity = self.xmlFile:getValue(key .. "#intensity", 1.0)
                map.idleValue = self.xmlFile:getValue(key .. "#idleValue", -1)
                map.toggleVisibility = self.xmlFile:getValue(key .. "#toggleVisibility", false)
                map.stateDelay = self.xmlFile:getValue(key .. "#stateDelay", 0.0)
                map.gpsAvailable = false
                self:resetDelayTimer(map)

                local hideNode = self.xmlFile:getValue(key .. "#hideNode", nil, self.components, self.i3dMappings)
                if hideNode ~= nil then
                    setVisibility(hideNode, false)
                end

                local showNode = self.xmlFile:getValue(key .. "#showNode", nil, self.components, self.i3dMappings)
                if showNode ~= nil then
                    setVisibility(showNode, true)
                end

                map.projection = {}
                map.projection.mode = self.xmlFile:getValue(key .. ".projection#mode", "PLAYER")
                map.projection.playerSize = self.xmlFile:getValue(key .. ".projection#playerSize", 1)
                map.projection.playerSizeMax = self.xmlFile:getValue(key .. ".projection#playerSizeMax", 1)
                map.projection.playerSizeMin = self.xmlFile:getValue(key .. ".projection#playerSizeMin", 0.001)
                map.projection.backgroundZoom = self.xmlFile:getValue(key .. ".projection#backgroundZoom", 1)
                map.projection.backgroundZoomMax = self.xmlFile:getValue(key .. ".projection#backgroundZoomMax", 20)
                map.projection.backgroundZoomMaxGPS = math.min(map.projection.backgroundZoomMax * 2, 60)
                map.projection.backgroundZoomMin = self.xmlFile:getValue(key .. ".projection#backgroundZoomMin", 1.35)
                map.projection.aspectRatio = self.xmlFile:getValue(key .. ".projection#aspectRatio", 1)

                table.insert(spec.maps, map)

                if g_currentMission:getIsClient() then
                    self:loadGPSMap(map)
                end
            end)

            self.xmlFile:iterate(RealGPS.HIDE_NODES_BASE_PATH:sub(1, -4), function(index, key)
                local hideNode = self.xmlFile:getValue(key .. "#node", nil, self.components, self.i3dMappings)
                local isDashboard = self.xmlFile:getValue(key .. "#isDashboard", false)

                if hideNode ~= nil then
                    table.insert(spec.hideNodes, {node = hideNode, isDashboard = isDashboard})
                    setVisibility(hideNode, false)
                end
            end)

            self.xmlFile:iterate(RealGPS.SHOW_NODES_BASE_PATH:sub(1, -4), function(index, key)
                local showNode = self.xmlFile:getValue(key .. "#node", nil, self.components, self.i3dMappings)

                if showNode ~= nil then
                    table.insert(spec.showNodes, showNode)
                    setVisibility(showNode, true)
                end
            end)
        end
    end
end

function RealGPS:onPostLoad(savegame)
    local spec = self.spec_realGPS

    if spec.hasRealGPSMap == true then
        for i, map in ipairs(spec.maps) do
            self:postLoadGPSMap(map, i, savegame)
        end
    end
end

function RealGPS:onPreInitComponentPlacement(savegame)
    local spec = self.spec_realGPS

    if spec.hasRealGPSMap then
        for _, hideNode in pairs(spec.hideNodes) do
            if hideNode.isDashboard then
                self:removeDashboardByNode(hideNode.node)
            end

            setVisibility(hideNode.node, false)
        end

        for _, showNode in pairs(spec.showNodes) do
            setVisibility(showNode, true)
        end
    end
end

function RealGPS:onRegisterDashboardValueTypes()
    local spec = self.spec_realGPS

    local playerPositionX = DashboardValueType.new("realGPS", "playerPositionX")
    playerPositionX:setValue(g_currentMission.hud.ingameMap, function(ingameMap) return ingameMap.normalizedPlayerPosX * g_currentMission.hud.ingameMap.worldSizeX end)
    self:registerDashboardValueType(playerPositionX)

    local playerPositionZ = DashboardValueType.new("realGPS", "playerPositionZ")
    playerPositionZ:setValue(g_currentMission.hud.ingameMap, function(ingameMap) return ingameMap.normalizedPlayerPosZ * g_currentMission.hud.ingameMap.worldSizeZ end)
    self:registerDashboardValueType(playerPositionZ)

    local playerRotation = DashboardValueType.new("realGPS", "playerRotation")
    playerRotation:setValue(self, function(self) 
        local rotation = math.deg(self:calculateRootWorldYAxisRotation())
        return rotation < 0 and rotation + 360 or rotation
    end)
    self:registerDashboardValueType(playerRotation)

    local gpsWorkingWidth = DashboardValueType.new("realGPS", "gpsWorkingWidth")
    gpsWorkingWidth:setValue(self, function(self)
        if self.spec_aiAutomaticSteering ~= nil and self.spec_aiAutomaticSteering.steeringFieldCourse ~= nil and self.spec_aiAutomaticSteering.steeringFieldCourse.fieldCourseSettings ~= nil then
            return self.spec_aiAutomaticSteering.steeringFieldCourse.fieldCourseSettings.implementWidth or 4
        else
            return 0
        end
    end)
    self:registerDashboardValueType(gpsWorkingWidth)
end

function RealGPS:onUpdate(dt)
    local spec = self.spec_realGPS

    if g_currentMission:getIsClient() and spec.hasRealGPSMap == true and self.rootNode ~= nil then

        if spec.isInSetupMode and #spec.maps > 1 then
            g_currentMission:addExtraPrintText(string.format(g_i18n:getText("info_currentSelectedMap"), spec.selectedMapIndex))
        end

        for i, map in ipairs(spec.maps) do
            self:setMotorDependentVisibility(map)

            local x, _, z = getWorldTranslation(self.rootNode)

            if map.rotatingMap then
                local mapXval = self:mapRange(g_currentMission.hud.ingameMap.normalizedPlayerPosX * g_currentMission.hud.ingameMap.worldSizeX, 0, g_currentMission.hud.ingameMap.worldSizeX, 0.25, 0.75)
                local mapYval = self:mapRange(g_currentMission.hud.ingameMap.normalizedPlayerPosZ * g_currentMission.hud.ingameMap.worldSizeZ, 0, g_currentMission.hud.ingameMap.worldSizeX, 0.75, 0.25)

                setShaderParameter(map.shaderObject , "mapProps", mapXval, mapYval, map.zoom, math.pi - self:calculateRootWorldYAxisRotation(), false)
                setShaderParameter(map.shaderObject , "playerProps", 0.5, 0.5, math.pi, map.projection.playerSize, false)
            else
                local mapXval = self:mapRange(g_currentMission.hud.ingameMap.normalizedPlayerPosX * g_currentMission.hud.ingameMap.worldSizeX, 0, g_currentMission.hud.ingameMap.worldSizeX, 0.25, 0.75)
                local mapYval = self:mapRange(g_currentMission.hud.ingameMap.normalizedPlayerPosZ * g_currentMission.hud.ingameMap.worldSizeZ, 0, g_currentMission.hud.ingameMap.worldSizeX, 0.75, 0.25)

                --print("mapXVal: " .. mapXval)
                --print("mapYVal " .. mapYval)
                --print("Zoom: " .. self.LRM.map.zoom)

                local mapX, playerX = self:calculateMapAndPlayerPosition(map, map.mapHighLimit, map.mapLowLimit, mapXval, map.zoom)
                local mapY, playerY = self:calculateMapAndPlayerPosition(map, map.mapHighLimit, map.mapLowLimit, mapYval, map.zoom)

                local rotation = self:calculateRootWorldYAxisRotation()

                --print("Player X:" .. playerX .. " PlayerY: " .. playerY .. " MapX: " .. mapX .. " MapY: " .. mapY)

                setShaderParameter(map.shaderObject , "playerProps", playerX, playerY, rotation, map.projection.playerSize, false)
                setShaderParameter(map.shaderObject , "mapProps", mapX, mapY, map.zoom, 0, false )	
            end

            
            local gpsAvailable = 0
            local gpsActive = 0
            local adjustDirection = 0
            local workingWidth = 4
            local segmentCoords = {
                x = 0,
                y = 0,
                z = 0,
                w = 0
            }

            if self.spec_aiAutomaticSteering ~= nil then
                local selectedAIMode = self:getAIModeSelection()

                if selectedAIMode == AIModeSelection.MODE.STEERING_ASSIST then
                    local steeringState = self:getAIAutomaticSteeringState()

                    map.gpsAvailable = true
                    gpsAvailable = 1

                    if steeringState == AIAutomaticSteering.STATE.ACTIVE then
                        gpsActive = 1
                    end

                    workingWidth = self:getAttacherToolWorkingWidth()

                    -- Aktives Segment der Spur
                    if workingWidth ~= nil and workingWidth ~= 0 and self.spec_aiAutomaticSteering.steeringFieldCourse ~= nil and self.spec_aiAutomaticSteering.steeringFieldCourse.currentSegment ~= nil then
                        local aiRootNode = self:getAIRootNode()
                        local course = self.spec_aiAutomaticSteering.steeringFieldCourse
                        local segment = self.spec_aiAutomaticSteering.steeringFieldCourse.currentSegment
                        if segment then
                            local tX, tZ, distanceToEnd = course:getSteeringTarget(aiRootNode, 0)
                            local wx, wy, wz = localToWorld(aiRootNode, tX, 0, tZ)
                            --DebugGizmo.renderAtPosition(wx, wy, wz, 0, 1, 0, 0, 0, 1, "AiRootNode")

                            -- Richtung Fahrzeug
                            local x, _, z = getWorldTranslation(aiRootNode)
                            local dirX, _, dirZ = localDirectionToWorld(aiRootNode, 0, 0, 1)
                            local vehDirLen = MathUtil.vector2Length(dirX, dirZ)
                            if vehDirLen > 0 then
                                dirX = dirX / vehDirLen
                                dirZ = dirZ / vehDirLen
                            end
                            -- Richtung Target
                            local toTargetX = wx - x
                            local toTargetZ = wz - z
                            local toTargetLen = MathUtil.vector2Length(toTargetX, toTargetZ)
                            if toTargetLen > 0 then
                                toTargetX = toTargetX / toTargetLen
                                toTargetZ = toTargetZ / toTargetLen
                            end
                            -- Kreuzprodukt für Vorzeichen (2D)
                            local cross = dirX * toTargetZ - dirZ * toTargetX
                            local distance = MathUtil.vector2Length(x - wx, z - wz)
                            if cross < 0 then
                                distance = -distance -- links
                            end
                            
                            if math.abs(distance) <= math.max(0.5, workingWidth * 0.1) then
                                distance = 0
                            end

                            adjustDirection = distance / workingWidth
                        end

                        if self.spec_aiAutomaticSteering.steeringFieldCourse.fieldCourseSettings ~= nil and self.spec_aiAutomaticSteering.steeringFieldCourse.fieldCourseSettings.implementWidth ~= nil then
                            workingWidth = self.spec_aiAutomaticSteering.steeringFieldCourse.fieldCourseSettings.implementWidth
                        end

                        local _, _, closestSegmentPartIndex = SteeringFieldCourse.getClosestPositionSegment(segment, x, z)
                        if closestSegmentPartIndex ~= nil then
                            local segmentPart = segment.positions[closestSegmentPartIndex]

                            if segmentPart ~= nil then
                                local closestSegmentPartX = segmentPart[1]
                                local closestSegmentPartZ = segmentPart[2]

                                local nextSegmentPart = segment.positions[closestSegmentPartIndex + 1]

                                if nextSegmentPart ~= nil then
                                    local nextSegmentPartX, nextSegmentPartZ = nextSegmentPart[1], nextSegmentPart[2]
                                    local segmentLength = MathUtil.vector2Length(closestSegmentPartX - nextSegmentPartX, closestSegmentPartZ - nextSegmentPartZ)

                                    if segmentLength > 0 then
                                        local function worldToUV(x, z, worldSizeX)
                                            local u = 0.5 + (x / (worldSizeX * 2))
                                            local v = 0.5 - (z / (worldSizeX * 2))
                                            return u, v
                                        end

                                        segmentCoords.x, segmentCoords.y = worldToUV(closestSegmentPartX, closestSegmentPartZ, g_currentMission.hud.ingameMap.worldSizeX)
                                        segmentCoords.z, segmentCoords.w = worldToUV(nextSegmentPartX, nextSegmentPartZ, g_currentMission.hud.ingameMap.worldSizeX)
                                    end

                                    local tX, tZ, distanceToEnd = course:getSteeringTarget(aiRootNode, 0)
                                    local _, wy, _ = localToWorld(aiRootNode, tX, 0, tZ)
                                    --DebugGizmo.renderAtPosition(closestSegmentPartX, wy, closestSegmentPartZ, 1, 0, 0, 0, 0, 1, "SegmentStart")
                                    --DebugGizmo.renderAtPosition(nextSegmentPartX, wy, nextSegmentPartZ, 0, 1, 0, 0, 0, 1, "SegmentEnd")
                                end
                            end
                        end
                    end
                else
                    map.gpsAvailable = false
                end
            end

            setShaderParameter(map.shaderObject, "gpsProps", gpsAvailable, gpsActive, adjustDirection, workingWidth, false)
            setShaderParameter(map.shaderObject, "gpsSegment", segmentCoords.x, segmentCoords.y, segmentCoords.z, segmentCoords.w, false)
        end
    end
end

function RealGPS:onUpdateTick(dt)
    local spec = self.spec_realGPS

    if g_currentMission:getIsClient() and self:getIsActive() and spec.hasRealGPSMap == true and self.rootNode ~= nil then
        for i, map in ipairs(spec.maps) do
            if map.startDelayTimer == true then
                map.delayTimer = map.delayTimer + dt

                if map.delayTimer >= map.stateDelay then
                    map.startDelayTimer = false
                    map.delayTimer = 0
                    map.delayTimerFinished = true
                end
            end

            if map.gpsAvailable == false and map.zoom > map.projection.backgroundZoomMax then
                local inputValue = (dt * -1) * 0.1
                self:setPlayerAndMapZoom(map, i, inputValue, map.projection.backgroundZoomMaxGPS)
            end
        end
    end
end

function RealGPS:resetDelayTimer(map)
    local spec = self.spec_realGPS

    map.startDelayTimer = false
    map.delayTimer = 0
    map.delayTimerFinished = false
end

function RealGPS:onDelete()
    local spec = self.spec_realGPS

    if g_currentMission:getIsClient() then
        if spec ~= nil and spec.maps ~= nil then
            for i, map in ipairs(spec.maps) do
                if map.sharedLoadRequestId ~= nil then
                    g_i3DManager:releaseSharedI3DFile(map.sharedLoadRequestId)
                    map.mapObject = nil
                    map.sharedLoadRequestId = nil
                end
            end
        end
    end
end

function RealGPS:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    local spec = self.spec_realGPS

    if g_currentMission:getIsClient() and spec.hasRealGPSMap == true then
        self:clearActionEventsTable(spec.actionEvents)

        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, "RGPS_SETUP_MODE", self, RealGPS.actionEvent_toggleSetupMode, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
            spec.actionEventId = actionEventId

            local _, actionEventId1 = self:addActionEvent(spec.actionEvents, "RGPS_TOGGLE_ROTATE_MODE", self, RealGPS.actionEvent_toggleRotateMode, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId1, GS_PRIO_NORMAL)
            spec.actionEventId1 = actionEventId1
			
			local _, actionEventId2 = self:addActionEvent(spec.actionEvents, "RGPS_AXIS_ZOOM", self, RealGPS.actionEvent_zoom, false, true, true, true, nil, nil, true)
			spec.actionEventId2 = actionEventId2
			g_inputBinding:setActionEventTextPriority(actionEventId2, GS_PRIO_NORMAL)

            local _, actionEventId3 = self:addActionEvent(spec.actionEvents, "RGPS_TOGGLE_SELECTED_MAP", self, RealGPS.actionEvent_toggleSelectedMap, false, true, false, true, nil)
            spec.actionEventId3 = actionEventId3
            g_inputBinding:setActionEventTextPriority(actionEventId3, GS_PRIO_NORMAL)

			RealGPS.updateActionEvents(self)
        end
    end
end

function RealGPS:updateActionEvents()
    local spec = self.spec_realGPS
    local isRotatingMap = spec.maps[spec.selectedMapIndex] ~= nil and spec.maps[spec.selectedMapIndex].rotatingMap or false

	g_inputBinding:setActionEventText(spec.actionEventId, spec.isInSetupMode and g_i18n:getText("input_RGPS_toggleSetupModeDisable") or g_i18n:getText("input_RGPS_toggleSetupModeEnable"))
    g_inputBinding:setActionEventText(spec.actionEventId1, isRotatingMap and g_i18n:getText("input_RGPS_toggleRotateModeDisable") or g_i18n:getText("input_RGPS_toggleRotateModeEnable"))
    g_inputBinding:setActionEventText(spec.actionEventId2, g_i18n:getText("input_RGPS_zoom"))
    g_inputBinding:setActionEventText(spec.actionEventId2, g_i18n:getText("input_RGPS_selectMap"))

    g_inputBinding:setActionEventActive(spec.actionEventId1, spec.isInSetupMode)
	g_inputBinding:setActionEventActive(spec.actionEventId2, spec.isInSetupMode)
    g_inputBinding:setActionEventActive(spec.actionEventId3, spec.isInSetupMode and #spec.maps > 1)
end

function RealGPS:loadGPSMap(map)
    local spec = self.spec_realGPS

    if map.linkNode ~= nil then
        local node, sharedLoadRequestId, failedReason = g_i3DManager:loadSharedI3DFile(RealGPS.MOD_SETTINGS_DIR .. "pdaMap/pdaMap.i3d", false, false)

        if failedReason ~= 0 then
            log:printError("Failed to load the PDA map. Error Id: " .. failedReason)
        end

        map.mapObject = node
        map.sharedLoadRequestId = sharedLoadRequestId

        link(map.linkNode, map.mapObject)
        local mapObject = getChildAt(map.linkNode, 0)
        map.shaderObject = getChildAt(mapObject, 0)

        spec.hasRealGPSMap = true
    end
end

function RealGPS:postLoadGPSMap(map, mapIndex, savegame)
    local zoom = 2
    local playerSize = 0.03125
    local rotateMap = map.projection.mode ~= "PLAYER"

    if savegame ~= nil then
        log:printDevInfo(string.format("Loading RealGPS settings for vehicle %s from savegame XML file", self:getName()), LoggingUtil.DEBUG_LEVELS.HIGH)
        log:printDevInfo("Map Index: " .. tostring(mapIndex - 1), LoggingUtil.DEBUG_LEVELS.HIGH)
        local key = savegame.key .. "." .. RealGPS.MOD_NAME .. ".realGPS.map(" .. (mapIndex - 1) .. ")"

        if hasXMLProperty(savegame.xmlFile.handle, key) then
            log:printDevInfo("Savegame XML Key: " .. key, LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Map Zoom: " .. tostring(zoom), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Map Zoom XML: " .. tostring(getXMLFloat(savegame.xmlFile.handle, key .. "#mapZoom")), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Player Size: " .. tostring(playerSize), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Player Size XML: " .. tostring(getXMLFloat(savegame.xmlFile.handle, key.. "#playerSize")), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Rotate Map: " .. tostring(rotateMap), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Rotate Map XML: " .. tostring(getXMLBool(savegame.xmlFile.handle, key .. "#rotateMap")), LoggingUtil.DEBUG_LEVELS.HIGH)

            zoom       = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, key .. "#mapZoom"),    zoom       )
            playerSize = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, key.. "#playerSize"), playerSize )
            rotateMap  = Utils.getNoNil(getXMLBool(savegame.xmlFile.handle, key .. "#rotateMap"),   rotateMap  )
        end
    end

    self:setMapZoom(map, zoom)
    self:setPlayerSize(map, playerSize)
    self:setRotateMap(map, rotateMap)

    if g_currentMission:getIsClient() then
        log:printDevInfo(string.format("Setting RealGPS Map props for vehicle %s", self:getName()), LoggingUtil.DEBUG_LEVELS.HIGH)
        setShaderParameter(map.shaderObject , "mapProps"    , 0.5, 0.5, map.zoom, 0, false)
        local aspectX = 1
        local aspectY = 1
        if map.projection.aspectRatio > 1 then
            aspectX = 1/map.projection.aspectRatio
        else
            aspectY = map.projection.aspectRatio
        end
        setShaderParameter(map.shaderObject , "screenProps" , aspectX, aspectY, g_currentMission.hud.ingameMap.worldSizeX, 0, false)
    end
end

function RealGPS:calculateRootWorldYAxisRotation()
    local object = self.rootNode
	local alpha, beta, gamma = getRotation(object)
	local yAxis = math.cos(gamma)*math.sin(beta)*math.cos(alpha) + math.sin(gamma)*math.sin(alpha)
	local xAxis = math.cos(beta)*math.cos(alpha)
	return(math.atan2(yAxis, xAxis))
end

function RealGPS:mapRange(value, fromMin, fromMax, toMin, toMax)
	local fromRange = fromMax - fromMin
	local toRange = toMax - toMin
	local scaledValue = (value - fromMin) / fromRange
	local mappedValue = toMin + (scaledValue * toRange)
	return mappedValue
end

function RealGPS:setMotorDependentVisibility(map)
    local spec = self.spec_realGPS

    if g_currentMission:getIsClient() then

        if (self.propertyState == VehiclePropertyState.SHOP_CONFIG or not self.getIsEntered) and map.shaderObject ~= nil then
            if map.toggleVisibility then
                setVisibility(map.shaderObject, false)
            else
                setShaderParameter(map.shaderObject, "lightControl", map.idleValue, 0, 0, 0, false)
            end
        else
            local shaderParameter = "lightControl"
            local x, y, z, w = getShaderParameter(map.shaderObject, shaderParameter)

            if spec.isMotorActiveDependent == true then
                if self:getIsMotorStarted() and map.stateDelay > 0 and not map.delayTimerFinished and not map.startDelayTimer then
                    map.startDelayTimer = true
                end

                if map.toggleVisibility then
                    if self:getIsMotorStarted() and x ~= map.intensity and (map.stateDelay == 0 or map.delayTimerFinished) then
                        setShaderParameter(map.shaderObject, shaderParameter, map.intensity, 0, 0, 0, false)
                        setVisibility(map.shaderObject, true)
                    elseif not self:getIsMotorStarted() and x ~= map.idleValue then
                        setShaderParameter(map.shaderObject, shaderParameter, map.idleValue, 0, 0, 0, false)
                        setVisibility(map.shaderObject, false)
                        self:resetDelayTimer(map)
                    end
                else
                    if self:getIsMotorStarted() and x ~= map.intensity and (map.stateDelay == 0 or map.delayTimerFinished) then
                        setShaderParameter(map.shaderObject, shaderParameter, map.intensity, 0, 0, 0, false)
                    elseif not self:getIsMotorStarted() and x ~= map.idleValue then
                        setShaderParameter(map.shaderObject, shaderParameter, map.idleValue, 0, 0, 0, false)
                        self:resetDelayTimer(map)
                    end
                end
            elseif spec.isIgnitionDependent == true then
                if self:getIsIgnitionActive() and map.stateDelay > 0 and not map.delayTimerFinished and not map.startDelayTimer then
                    map.startDelayTimer = true
                end

                if map.toggleVisibility then
                    if self:getIsIgnitionActive() and x ~= map.intensity and (map.stateDelay == 0 or map.delayTimerFinished) then
                        setShaderParameter(map.shaderObject, shaderParameter, map.intensity, 0, 0, 0, false)
                        setVisibility(map.shaderObject, true)
                    elseif (not self:getIsIgnitionActive() and x ~= map.idleValue) or spec.isLoading then
                        setShaderParameter(map.shaderObject, shaderParameter, map.idleValue, 0, 0, 0, false)
                        setVisibility(map.shaderObject, false)
                        self:resetDelayTimer(map)
                        spec.isLoading = false
                    end
                else
                    if self:getIsIgnitionActive() and x ~= map.intensity and (map.stateDelay == 0 or map.delayTimerFinished) then
                        setShaderParameter(map.shaderObject, shaderParameter, map.intensity, 0, 0, 0, false)
                    elseif (not self:getIsIgnitionActive() and x ~= map.idleValue) or spec.isLoading then
                        setShaderParameter(map.shaderObject, shaderParameter, map.idleValue, 0, 0, 0, false)
                        self:resetDelayTimer(map)
                        spec.isLoading = false
                    end
                end
            elseif x ~= map.intensity then
                setShaderParameter(map.shaderObject, shaderParameter, map.intensity, 0, 0, 0, false)
            end
        end
    end
end

function RealGPS:calculateMapAndPlayerPosition(map, highLimit, lowLimit, value, zoom)
    local spec = self.spec_realGPS

	if value <= lowLimit then
		return lowLimit, (value * zoom - 0.5 * zoom / 2)
	elseif value >= highLimit then
		return highLimit, self:mapRange(value, map.mapHighLimit, 0.75, 0.5, 1)
	else
		return value, 0.5
	end
end

function RealGPS:setMapZoom(map, value)
    local spec = self.spec_realGPS
    map.zoom = value
    log:printDevInfo("Set map zoom to " .. tostring(map.zoom), LoggingUtil.DEBUG_LEVELS.HIGH)

    if g_currentMission:getIsClient() then
        map.mapLowLimit  = 0.5 / map.zoom + 0.25
        map.mapHighLimit = 1 - map.mapLowLimit
    end
end

function RealGPS:setPlayerSize(map, value)
    local spec = self.spec_realGPS
    map.projection.playerSize = value
    log:printDevInfo("Set player size to " .. tostring(map.projection.playerSize), LoggingUtil.DEBUG_LEVELS.HIGH)

    if g_currentMission:getIsClient() then
        if map.rotatingMap then
            setShaderParameter(map.shaderObject , "playerProps", 0.5, 0.5, math.pi, map.projection.playerSize, false)
        end
    end
end

function RealGPS:setRotateMap(map, value)
    local spec = self.spec_realGPS
    map.rotatingMap = value
    log:printDevInfo("Set map rotation to " .. tostring(map.rotatingMap), LoggingUtil.DEBUG_LEVELS.HIGH)

    if g_currentMission:getIsClient() then
        if map.rotatingMap then
            setShaderParameter(map.shaderObject , "playerProps", 0.5, 0.5, math.pi, map.projection.playerSize, false)
        end
    end
end

function RealGPS:actionEvent_toggleSetupMode(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_realGPS
	spec.isInSetupMode = not spec.isInSetupMode
	SetupModeEvent.sendEvent(self, spec.isInSetupMode, false)
	RealGPS.updateActionEvents(self)
end

function RealGPS:actionEvent_toggleRotateMode(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_realGPS
	if spec.isInSetupMode then
		if spec.controls.allowRotation then
            for i, map in ipairs(spec.maps) do
                if i == spec.selectedMapIndex then
                    self:setRotateMap(map, not map.rotatingMap)
                    RealGPS.updateActionEvents(self)

                    RotationModeEvent.sendEvent(self, i, map.rotatingMap, false)
                end
            end
		end
	end
end

function RealGPS:actionEvent_zoom(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_realGPS
    if spec.isInSetupMode then
        for i, map in ipairs(spec.maps) do
            if i == spec.selectedMapIndex then
                local maxZoom = map.gpsAvailable and map.projection.backgroundZoomMaxGPS or map.projection.backgroundZoomMax
                self:setPlayerAndMapZoom(map, i, inputValue, maxZoom)
            end
        end
    end
end

function RealGPS:actionEvent_toggleSelectedMap(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_realGPS

    if spec.isInSetupMode then
        local maxMapIndex = #spec.maps
        
        spec.selectedMapIndex = (spec.selectedMapIndex + 1) % (maxMapIndex + 1)

        if spec.selectedMapIndex == 0 then
            spec.selectedMapIndex = 1
        end
    end
end

function RealGPS:setPlayerAndMapZoom(map, mapId, inputValue, maxZoom)
    local spec = self.spec_realGPS
    local newZoom = map.zoom + inputValue * 0.1
    local allowPlayerSizing = true

    if newZoom < map.projection.backgroundZoomMin then
        newZoom = map.projection.backgroundZoomMin
        allowPlayerSizing = false
    elseif newZoom > maxZoom then
        newZoom = maxZoom
        allowPlayerSizing = false
    end

    if allowPlayerSizing then
        local newSize = map.projection.playerSize + ((inputValue * 0.01) / (maxZoom - map.projection.backgroundZoomMin) / 2)

        if newSize < map.projection.playerSizeMin then
            newSize = map.projection.playerSizeMin
        elseif newSize > map.projection.playerSizeMax then
            newSize = map.projection.playerSizeMax
        end
        self:setPlayerSize(map, newSize)
    end

    self:setMapZoom(map, newZoom)

    MapZoomEvent.sendEvent(self, mapId, map.zoom, map.projection.playerSize, false)
end

function RealGPS:getIsIgnitionActive()
    if g_ignitionLockManager:getIsAvailable() then
        local ignitionState = g_ignitionLockManager:getState()
        if ignitionState == IgnitionLockState.OFF then
            return false
        elseif ignitionState == IgnitionLockState.IGNITION or ignitionState == IgnitionLockState.START then
            return true
        end
    else
        local motorState = self:getMotorState()
        if motorState == MotorState.ON or motorState == MotorState.STARTING or motorState == MotorState.IGNITION then
            return true
        else
            return false
        end
    end
end

function RealGPS:loadDefaultVehicles()
    log:printDevInfo("Loading default vehicles", LoggingUtil.DEBUG_LEVELS.HIGH)
    RealGPS.defaultVehicles = {}
    local defaultVehiclesXML = XMLFile.loadIfExists("realGPSDefaultVehicles", RealGPS.DEFAULT_VEHICLE_XML, RealGPS.DEFAULT_VEHICLE_SCHEMA)

    if defaultVehiclesXML ~= nil then
        defaultVehiclesXML:iterate(RealGPS.DEFAULT_VEHICLE_BASE_PATH:sub(1, -4), function (index, key)
            local filename = defaultVehiclesXML:getString(key .. "#filename")
            filename = Utils.getFilename(filename)

            if RealGPS.defaultVehicles[filename] == nil then
                RealGPS.defaultVehicles[filename] = {}
                local defaultVehicle = RealGPS.defaultVehicles[filename]

                defaultVehicle.isMotorActiveDependent = defaultVehiclesXML:getValue(key .. "#isMotorActiveDependent", false)
                defaultVehicle.isIgnitionDependent = defaultVehiclesXML:getValue(key .. "#isIgnitionDependent", false)

                defaultVehicle.controls = {}
                defaultVehicle.controls.allow = defaultVehiclesXML:getValue(key .. ".controls#allow", true)
                defaultVehicle.controls.allowRotation = defaultVehiclesXML:getValue(key .. ".controls#allowRotation", true)
                defaultVehicle.controls.allowZoom = defaultVehiclesXML:getValue(key .. ".controls#allowZoom", true)

                --defaultVehicle.map = {}
                defaultVehicle.maps = {}
                defaultVehicle.hideNodes = {}
                defaultVehicle.showNodes = {}

                defaultVehiclesXML:iterate(key .. ".map", function(index, key2)
                    local map = {}

                    map.rootNode = defaultVehiclesXML:getValue(key2 .. "#rootNode", nil)
                    map.hideNode = defaultVehiclesXML:getValue(key2 .. "#hideNode", nil)
                    map.showNode = defaultVehiclesXML:getValue(key2 .. "#showNode", nil)
                    map.translation = defaultVehiclesXML:getValue(key2 .. "#translation", "0 0 0", true)
                    map.rotation = defaultVehiclesXML:getValue(key2 .. "#rotation", "0 0 0", true)
                    map.scale = defaultVehiclesXML:getValue(key2 .. "#scale", "1 1 1", true)
                    map.intensity = defaultVehiclesXML:getValue(key2 .. "#intensity", 1.0)
                    map.idleValue = defaultVehiclesXML:getValue(key2 .. "#idleValue", -1)
                    map.toggleVisibility = defaultVehiclesXML:getValue(key2 .. "#toggleVisibility", false)
                    map.stateDelay = defaultVehiclesXML:getValue(key2 .. "#stateDelay", 0.0)
                    map.gpsAvailable = false

                    local projection = {}
                    projection.mode = defaultVehiclesXML:getValue(key2 .. ".projection#mode", "PLAYER")
                    projection.playerSize = defaultVehiclesXML:getValue(key2 .. ".projection#playerSize", 1)
                    projection.playerSizeMax = defaultVehiclesXML:getValue(key2 .. ".projection#playerSizeMax", 0.5)
                    projection.playerSizeMin = defaultVehiclesXML:getValue(key2 .. ".projection#playerSizeMin", 0.001)
                    projection.backgroundZoom = defaultVehiclesXML:getValue(key2 .. ".projection#backgroundZoom", 1)
                    projection.backgroundZoomMax = defaultVehiclesXML:getValue(key2 .. ".projection#backgroundZoomMax", 20)
                    projection.backgroundZoomMaxGPS = math.min(projection.backgroundZoomMax * 2, 60)
                    projection.backgroundZoomMin = defaultVehiclesXML:getValue(key2 .. ".projection#backgroundZoomMin", 1.35)
                    projection.aspectRatio = defaultVehiclesXML:getValue(key2 .. ".projection#aspectRatio", 1)

                    map.projection = projection
                    table.insert(defaultVehicle.maps, map)
                end)

                defaultVehiclesXML:iterate(key .. ".hideNodes.hideNode", function(index, key2)
                    local hideNode = defaultVehiclesXML:getValue(key2 .. "#node", nil)
                    if hideNode ~= nil then
                        
                        local hideNodeData = {
                            node = hideNode,
                            isDashboard = defaultVehiclesXML:getValue(key2 .. "#isDashboard", false)
                        }

                        table.insert(defaultVehicle.hideNodes, hideNodeData)
                    end
                end)

                defaultVehiclesXML:iterate(key .. ".showNodes.showNode", function(index, key2)
                    local showNode = defaultVehiclesXML:getValue(key2 .. "#node", nil)
                    if showNode ~= nil then
                        table.insert(defaultVehicle.showNodes, showNode)
                    end
                end)
            end
        end)
        log:printDevInfo(string.format("Loaded %d default vehicles", #RealGPS.defaultVehicles), LoggingUtil.DEBUG_LEVELS.HIGH)
    end
end

function RealGPS:onDashboardCompoundLoaded(superFunc, i3dNode, failedReason, args)
    local dashboardXMLFile = args.dashboardXMLFile
    local compound = args.compound
    local compoundKey = args.compoundKey

    if i3dNode ~= 0 then
        local components = {}
        for i=1, getNumOfChildren(i3dNode) do
            table.insert(components, {node=getChildAt(i3dNode, i - 1)})
        end

        compound.i3dMappings = {}
        I3DUtil.loadI3DMapping(dashboardXMLFile, "dashboardCompounds", components, compound.i3dMappings, nil)

        dashboardXMLFile:iterate(compoundKey .. ".realGPS", function(index2, key2)
            if self.configurations["realGPS"] ~= nil and self.configurations["realGPS"] == 2 then
                self:loadDashboardCompoundRealGPS(i3dNode, components, compound.i3dMappings, dashboardXMLFile, key2)
            end
        end)
    end

    superFunc(self, i3dNode, failedReason, args)
    RealGPS.updateActionEvents(self)
end

function RealGPS:loadDashboardCompoundRealGPS(i3dNode, components, i3dMappings, dashboardXMLFile, realGPSKey)
    local spec = self.spec_realGPS

    spec.isIgnitionDependent = dashboardXMLFile:getValue(realGPSKey.. "#isIgnitionDependent", false)
    spec.isMotorActiveDependent = dashboardXMLFile:getValue(realGPSKey .. "#isMotorActiveDependent", false)

    if spec.isIgnitionDependent == true and spec.isMotorActiveDependent == true then
        spec.isMotorActiveDependent = false
    end

    spec.isLoading = true
    spec.controls = spec.controls or {}

    local numControls = 0
    for _, _ in pairs(spec.controls) do
        numControls = numControls + 1
    end

    if self.xmlFile:hasProperty(RealGPS.BASE_PATH) and self.xmlFile:hasProperty(RealGPS.CONTROLS_BASE_PATH) and self.xmlFile:hasProperty(RealGPS.MAP_BASE_PATH) and spec.controls ~= nil and numControls > 0 then
        log:printDevWarning("Vehicle " .. self.xmlFile.filename .. " has RealGPS properties defined in the vehicle XML file. These will override the properties defined in the loaded dashboard compound '" .. dashboardXMLFile.filename .. "'.", LoggingUtil.DEBUG_LEVELS.HIGH)
    end

    if #spec.controls == 0 then
        spec.controls.allow = dashboardXMLFile:getValue(realGPSKey .. ".controls#allow", true)
        spec.controls.allowRotation = dashboardXMLFile:getValue(realGPSKey .. ".controls#allowRotation", true)
        spec.controls.allowZoom = dashboardXMLFile:getValue(realGPSKey .. ".controls#allowZoom", true)
    end

    dashboardXMLFile:iterate(realGPSKey .. ".map", function(index, key)
        local map = {}
        map.linkNode = dashboardXMLFile:getValue(key .. "#linkNode", nil, components, i3dMappings)
        map.intensity = dashboardXMLFile:getValue(key .. "#intensity", 1.0)
        map.idleValue = dashboardXMLFile:getValue(key .. "#idleValue", -1)
        map.toggleVisibility = dashboardXMLFile:getValue(key .. "#toggleVisibility", false)
        map.stateDelay = dashboardXMLFile:getValue(key .. "#stateDelay", 0.0)
        map.gpsAvailable = false
        self:resetDelayTimer(map)

        local hideNode = dashboardXMLFile:getValue(key .. "#hideNode", nil, components, i3dMappings)
        if hideNode ~= nil then
            setVisibility(hideNode, false)
        end

        local showNode = dashboardXMLFile:getValue(key .. "#showNode", nil, components, i3dMappings)
        if showNode ~= nil then
            setVisibility(showNode, true)
        end

        map.projection = {}
        map.projection.mode = dashboardXMLFile:getValue(key .. ".projection#mode", "PLAYER")
        map.projection.playerSize = dashboardXMLFile:getValue(key .. ".projection#playerSize", 1)
        map.projection.playerSizeMax = dashboardXMLFile:getValue(key .. ".projection#playerSizeMax", 1)
        map.projection.playerSizeMin = dashboardXMLFile:getValue(key .. ".projection#playerSizeMin", 0.001)
        map.projection.backgroundZoom = dashboardXMLFile:getValue(key .. ".projection#backgroundZoom", 1)
        map.projection.backgroundZoomMax = dashboardXMLFile:getValue(key .. ".projection#backgroundZoomMax", 20)
        map.projection.backgroundZoomMaxGPS = math.min(map.projection.backgroundZoomMax * 2, 60)
        map.projection.backgroundZoomMin = dashboardXMLFile:getValue(key .. ".projection#backgroundZoomMin", 1.35)
        map.projection.aspectRatio = dashboardXMLFile:getValue(key .. ".projection#aspectRatio", 1)

        table.insert(spec.maps, map)
        local mapIndex = #spec.maps

        if g_currentMission:getIsClient() then
            self:loadGPSMap(map)
            self:postLoadGPSMap(map, mapIndex, self.savegame)

            if #spec.actionEvents == 0 then
                RealGPS.onRegisterActionEvents(self, self:getIsActiveForInput(), self:getIsActiveForInput(true))
            end
        end
    end)

    dashboardXMLFile:iterate(realGPSKey .. ".hideNodes.hideNode", function(index, key)
        local hideNode = dashboardXMLFile:getValue(key .. "#node", nil, components, i3dMappings)
        if hideNode ~= nil then
            local isDashboard = dashboardXMLFile:getValue(key .. "#isDashboard", false)

            local hideNodeData = {
                node = hideNode,
                isDashboard = isDashboard
            }

            table.insert(spec.hideNodes, hideNodeData)

            if not isDashboard then
                setVisibility(hideNode, false)
            end
        end
    end)

    dashboardXMLFile:iterate(realGPSKey .. ".showNodes.showNode", function(index, key)
        local showNode = dashboardXMLFile:getValue(key .. "#node", nil, components, i3dMappings)
        if showNode ~= nil then
            table.insert(spec.showNodes, showNode)
            setVisibility(showNode, true)
        end
    end)
end

function RealGPS:loadDashboardsFromXML(superFunc, xmlFile, key, dashboardValueType, components, i3dMappings, parentNode)
    local success = superFunc(self, xmlFile, key, dashboardValueType, components, i3dMappings, parentNode)

    if key:find("dashboardCompound") then
        key = key .. ".realGPS.dashboards"

        if xmlFile:hasProperty(key) then
            success = superFunc(self, xmlFile, key, dashboardValueType, components, i3dMappings, parentNode)
        end
    end

    return success
end

function RealGPS:removeDashboardByNode(node)
    local spec_dashboard = self.spec_dashboard

    if spec_dashboard ~= nil then
        local combinedDashboards = {spec_dashboard.groupDashboards, spec_dashboard.tickDashboards, spec_dashboard.criticalDashboards}

        for _, dashboardGroup in pairs(combinedDashboards) do
            for i = #dashboardGroup, 1, -1 do
                local dashboard = dashboardGroup[i]
                if dashboard.node == node then
                    table.remove(dashboardGroup, i)
                    spec_dashboard.numDashboards = spec_dashboard.numDashboards - 1
                end
            end
        end
    end
end